/*
 * Decompiled with CFR 0.152.
 */
package math;

import biomart.BiomartConnector;
import biomart.SpeciesSettings;
import gui.Logger;
import io.BamFileReader;
import io.BedFileReader;
import io.BowtieFileReader;
import io.SamFileReader;
import io.TagFileReader;
import io.TagFileReaderListener;
import io.TagalignFileReader;
import io.WiggleFileReader;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Random;
import math.LocusInfo;
import math.PositionInfo;
import math.ProfilerData;
import math.ProfilerDataStatistics;
import math.ProfilerListener;
import math.Sort;
import settings.ExperimentSettings;

public class Profiler
implements TagFileReaderListener {
    private ProfilerData data;
    private ExperimentSettings expSettings;
    private ArrayList<ProfilerListener> listeners;
    private double[] histogram;
    private boolean forceStop = false;
    private int cores;
    private TagFileReader tagReader;

    public Profiler(ExperimentSettings expSettings, int chromosomes) {
        this.expSettings = expSettings;
        this.listeners = new ArrayList();
        this.data = new ProfilerData(chromosomes);
        this.cores = Runtime.getRuntime().availableProcessors();
    }

    public void forceStop() {
        this.forceStop = true;
        if (this.tagReader != null) {
            this.tagReader.forceStopReading();
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).forceStop();
            ++l;
        }
    }

    public void setProfilerData(ProfilerData data) {
        this.data = data;
    }

    public void readPositionsData() throws IOException {
        if (this.expSettings.getParams().isLociLoad()) {
            this.loadLociList();
        } else {
            this.getDataFromDataBase();
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).dataFromDataBaseReceived(this.data);
            ++l;
        }
    }

    public void readTagData() throws IOException {
        this.readTags();
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).tagsReadingFinished(this.data);
            ++l;
        }
    }

    public ProfilerData getData() {
        return this.data;
    }

    public void buildBindingProfile() {
        if (this.expSettings.getParams().isFilterShortLoci()) {
            this.filterShortLoci();
        }
        if (this.expSettings.getParams().isLeaveNoNearLoci()) {
            this.filterAllLoci();
        } else if (this.expSettings.getParams().isLeaveRandomNearLoci()) {
            this.filterRandomLoci();
        } else if (this.expSettings.getParams().isLeaveMiddleNearLoci()) {
            this.filterMiddleLoci();
        }
        if (this.forceStop) {
            return;
        }
        if (this.expSettings.getParams().isPrefilterLoci()) {
            float[][] allRemovedTags = this.prefilterLoci();
            int[] allRemoved = new int[allRemovedTags.length];
            int i = 0;
            while (i < allRemoved.length) {
                allRemoved[i] = (int)allRemovedTags[i][1];
                ++i;
            }
            int l = 0;
            while (l < this.listeners.size()) {
                this.listeners.get(l).prefilteringLociFinished(this.data, allRemoved);
                ++l;
            }
        }
        this.assignTagsToLoci(this.expSettings.getParams().getLocusLength(), this.expSettings.getParams().getLocusOffset());
        ProfilerDataStatistics.computeStatistics(this.data);
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).tagsMatchingFinished(this.data);
            ++l;
        }
        if (this.forceStop) {
            return;
        }
        if (this.expSettings.getParams().isFilterMultipleAssignments()) {
            float[] allRemoved = this.filterMultipleAssignments();
            ProfilerDataStatistics.computeStatistics(this.data);
            int l2 = 0;
            while (l2 < this.listeners.size()) {
                this.listeners.get(l2).filteringAssignmentsFinished(this.data, allRemoved);
                ++l2;
            }
        }
        if (this.forceStop) {
            return;
        }
        if (this.expSettings.getParams().isFilterLowTagLoci()) {
            float[][] allRemoved = this.filterLowTagsLoci(this.expSettings.getParams().getMinLociTagCount(), true);
            ProfilerDataStatistics.computeStatistics(this.data);
            int l3 = 0;
            while (l3 < this.listeners.size()) {
                this.listeners.get(l3).filteringLowTagsLociFinished(this.data, allRemoved);
                ++l3;
            }
        }
        if (this.expSettings.getParams().isFilterLowTagPositions()) {
            float[][] allRemoved = this.filterLowTagsPositions();
            ProfilerDataStatistics.computeStatistics(this.data);
            int l4 = 0;
            while (l4 < this.listeners.size()) {
                this.listeners.get(l4).filteringLowTagsPositionsFinished(this.data, allRemoved);
                ++l4;
            }
        }
        if (this.expSettings.getParams().isFilterMultiTagLoci()) {
            float[][] allRemoved = this.filterMultiTagsLoci();
            ProfilerDataStatistics.computeStatistics(this.data);
            int l5 = 0;
            while (l5 < this.listeners.size()) {
                this.listeners.get(l5).filteringMultiTagsLociFinished(this.data, allRemoved);
                ++l5;
            }
        }
        if (this.expSettings.getParams().isFilterMultiTagPositions()) {
            float[][] allRemoved = this.filterMultiTagsPositions();
            ProfilerDataStatistics.computeStatistics(this.data);
            int l6 = 0;
            while (l6 < this.listeners.size()) {
                this.listeners.get(l6).filteringMultiTagsPositionsFinished(this.data, allRemoved);
                ++l6;
            }
        }
        if (this.expSettings.getParams().isLociSave()) {
            this.saveLociList();
        }
        if (!this.expSettings.getParams().isAssignmentsCountNormalisation()) {
            this.generateHistogram();
        } else {
            this.generateNormalizedHistogram();
        }
    }

    private void loadLociList() {
        File f = new File(this.expSettings.getParams().getLociLoadFile());
        try {
            String line;
            BufferedReader br = new BufferedReader(new FileReader(f));
            while ((line = br.readLine()) != null) {
                if (line.length() == 0 || line.charAt(0) == '#') continue;
                String[] lineTab = line.split("\t");
                int ch = Integer.parseInt(lineTab[0]);
                LocusInfo locus = new LocusInfo();
                locus.setCoordinate(Integer.parseInt(lineTab[1]));
                locus.setLength(Integer.parseInt(lineTab[2]));
                locus.setName(lineTab[3]);
                locus.setId(lineTab[4]);
                locus.setDescription(lineTab[5]);
                locus.setStrand(Byte.parseByte(lineTab[6]));
                this.data.addLocus(ch, locus);
            }
            br.close();
        }
        catch (Exception e) {
            Logger.writeExceptionLog(e);
        }
    }

    private void saveLociList() {
        File f = new File(this.expSettings.getParams().getLociSaveFile());
        try {
            if (f.exists()) {
                f.delete();
            }
            f.createNewFile();
            BufferedWriter br = new BufferedWriter(new FileWriter(f));
            br.write("#chromosome\tcoordinate\tlength\tname\tid\tdescription\tstrand");
            br.newLine();
            int i = 0;
            while (i < this.data.getChromosomes()) {
                LocusInfo[] loci = this.data.getLoci(i);
                int l = 0;
                while (l < loci.length) {
                    br.write(String.valueOf(i) + "\t" + loci[l].toString());
                    br.newLine();
                    ++l;
                }
                ++i;
            }
            br.close();
        }
        catch (Exception e) {
            Logger.writeExceptionLog(e);
        }
    }

    private void generateHistogram() {
        this.histogram = new double[this.expSettings.getParams().getLocusLength()];
        int offset = this.expSettings.getParams().getLocusOffset() - this.expSettings.getParams().getLocusLength() / 2;
        int i = 0;
        while (i < this.data.getChromosomes()) {
            PositionInfo[] positions = this.data.getMatchedPositions(i);
            int j = 0;
            while (j < positions.length) {
                try {
                    int n = positions[j].getLociDistance() - offset;
                    this.histogram[n] = this.histogram[n] + (double)positions[j].getTagCount();
                }
                catch (IndexOutOfBoundsException e) {
                    e.printStackTrace();
                    System.out.println("###D" + positions[j].getLociDistance() + "####");
                    System.out.println("###O" + offset + "####");
                }
                ++j;
            }
            ++i;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).histogramGenerated(this.histogram);
            ++l;
        }
    }

    private void generateNormalizedHistogram() {
        this.histogram = new double[this.expSettings.getParams().getLocusLength()];
        int offset = this.expSettings.getParams().getLocusOffset() - this.expSettings.getParams().getLocusLength() / 2;
        int currentPosition = -1;
        float norm = 0.0f;
        int index = 0;
        int i = 0;
        while (i < this.data.getChromosomes()) {
            Object[] positions = this.data.getMatchedPositions(i);
            Arrays.sort(positions);
            int j = 0;
            while (j < positions.length) {
                if (((PositionInfo)positions[j]).getLociCoordinate() != currentPosition) {
                    norm = ((PositionInfo)positions[j]).getTagCount();
                    index = 1;
                    currentPosition = ((PositionInfo)positions[j]).getLociCoordinate();
                    while (j + index < positions.length && ((PositionInfo)positions[j + index]).getLociCoordinate() == currentPosition) {
                        norm += ((PositionInfo)positions[j + index]).getTagCount();
                        ++index;
                    }
                }
                int n = ((PositionInfo)positions[j]).getLociDistance() - offset;
                this.histogram[n] = this.histogram[n] + 1.0 * (double)((PositionInfo)positions[j]).getTagCount() / (double)norm;
                ++j;
            }
            ++i;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).histogramGenerated(this.histogram);
            ++l;
        }
    }

    private void getDataFromDataBase() throws IOException {
        ArrayList<String[]> list;
        String[] attributes;
        HashMap<String, ArrayList<String>> filterMap = new HashMap<String, ArrayList<String>>(2);
        if (this.expSettings.getParams().isFilterLociTypes()) {
            filterMap.put("biotype", this.expSettings.getParams().getLociTypes());
        }
        if (this.expSettings.getParams().isFilterGOTerm()) {
            String line;
            File f = new File(this.expSettings.getParams().getGOTermsFile());
            ArrayList<String> terms = new ArrayList<String>();
            BufferedReader br = new BufferedReader(new FileReader(f));
            while ((line = br.readLine()) != null) {
                if ((line = line.trim()).length() <= 0) continue;
                terms.add(line);
            }
            filterMap.put("go_parent_term", terms);
        }
        if (this.expSettings.getParams().isCheckStrand()) {
            attributes = new String[]{"chromosome_name", "strand", this.expSettings.getParams().getMainAttribute(), this.expSettings.getParams().getSecondaryAttribute(), "external_gene_id", "ensembl_gene_id", "gene_biotype", "transcript_biotype"};
            list = BiomartConnector.getData(attributes, this.expSettings.getParams().getSpeciesId(), filterMap);
            int i = 0;
            while (i < list.size()) {
                int ch = -1;
                int str = 0;
                ch = SpeciesSettings.getChromosomeId(this.expSettings.getParams().getSpeciesId(), list.get(i)[0]);
                if (ch >= 0) {
                    LocusInfo locus = new LocusInfo();
                    str = Integer.parseInt(list.get(i)[1]);
                    locus.setStrand((byte)str);
                    if (str == 1) {
                        locus.setCoordinate(Integer.parseInt(list.get(i)[2]));
                    } else if (str == -1) {
                        locus.setCoordinate(Integer.parseInt(list.get(i)[3]));
                    }
                    locus.setLength(Math.abs(Integer.parseInt(list.get(i)[2]) - Integer.parseInt(list.get(i)[3])));
                    locus.setName(list.get(i)[4]);
                    locus.setId(list.get(i)[5]);
                    locus.setDescription(list.get(i)[6]);
                    if (list.get(i)[6].equals(list.get(i)[7])) {
                        this.data.addLongestLocus(ch, locus);
                    }
                }
                ++i;
            }
        } else {
            attributes = new String[]{"chromosome_name", this.expSettings.getParams().getMainAttribute(), "external_gene_id", "ensembl_gene_id", "gene_biotype", "transcript_biotype"};
            list = BiomartConnector.getData(attributes, this.expSettings.getParams().getSpeciesId(), filterMap);
            int i = 0;
            while (i < list.size()) {
                int ch = -1;
                ch = SpeciesSettings.getChromosomeId(this.expSettings.getParams().getSpeciesId(), list.get(i)[0]);
                if (ch >= 0) {
                    LocusInfo locus = new LocusInfo();
                    locus.setCoordinate(Integer.parseInt(list.get(i)[1]));
                    locus.setName(list.get(i)[2]);
                    locus.setId(list.get(i)[3]);
                    locus.setDescription(list.get(i)[4]);
                    if (list.get(i)[4].equals(list.get(i)[5])) {
                        this.data.addLongestLocus(ch, locus);
                    }
                }
                ++i;
            }
        }
    }

    private void readTags() throws IOException {
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).tagsReadingStart(100L);
            ++l;
        }
        this.tagReader = this.expSettings.getBowtieFile().toLowerCase().endsWith(".sam") ? new SamFileReader() : (this.expSettings.getBowtieFile().toLowerCase().endsWith(".bam") ? new BamFileReader() : (this.expSettings.getBowtieFile().toLowerCase().endsWith(".wig") ? new WiggleFileReader() : (this.expSettings.getBowtieFile().toLowerCase().endsWith(".bed") ? new BedFileReader() : (this.expSettings.getBowtieFile().toLowerCase().endsWith(".tagalign") ? new TagalignFileReader() : new BowtieFileReader()))));
        this.tagReader.addListener(this);
        this.tagReader.readTags(this.expSettings, this.data);
    }

    private int[] filterShortLoci() {
        int[] allRemoved = new int[this.data.getChromosomes()];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        ShortLociFilterThread[] shortLociFilterThread = new ShortLociFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                shortLociFilterThread[threadCount] = new ShortLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(shortLociFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[shortLociFilterThread[j].getChromosome()] = shortLociFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                shortLociFilterThread[threadCount] = new ShortLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(shortLociFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[shortLociFilterThread[i2].getChromosome()] = shortLociFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).filteringShortLociFinished(this.data, allRemoved);
            ++l;
        }
        return allRemoved;
    }

    private int filterShortLociInChromosome(int chromosome) {
        if (this.data.getLociCount(chromosome) < 2) {
            return 0;
        }
        int removed = 0;
        LocusInfo[] loci = this.data.getLoci(chromosome);
        int i = 0;
        while (i < loci.length) {
            if (loci[i].getLength() < this.expSettings.getParams().getMinLociLength()) {
                loci[i].setFiltered(true);
                ++removed;
            }
            ++i;
        }
        return removed;
    }

    private float[][] prefilterLoci() {
        this.assignTagsToLoci(this.expSettings.getParams().getPrefilterLociLength(), this.expSettings.getParams().getPrefilterLociOffset());
        float[][] allRemoved = this.filterLowTagsLoci(this.expSettings.getParams().getPrefilterLociTags(), false);
        int i = 0;
        while (i < this.data.getChromosomes()) {
            LocusInfo[] loci = this.data.getAllLoci(i);
            int l = 0;
            while (l < loci.length) {
                loci[l].setTagCount(0.0f);
                loci[l].setMatched(false);
                ++l;
            }
            ++i;
        }
        this.data.restorePositions();
        return allRemoved;
    }

    private int[] filterAllLoci() {
        int[] allRemoved = new int[this.data.getChromosomes()];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        AllLociFilterThread[] allLociFilterThread = new AllLociFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                allLociFilterThread[threadCount] = new AllLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(allLociFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[allLociFilterThread[j].getChromosome()] = allLociFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                allLociFilterThread[threadCount] = new AllLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(allLociFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[allLociFilterThread[i2].getChromosome()] = allLociFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).filteringNearLociFinished(this.data, allRemoved);
            ++l;
        }
        return allRemoved;
    }

    private int filterAllLociInChromosome(int chromosome) {
        if (this.data.getLociCount(chromosome) < 2) {
            return 0;
        }
        int removed = 0;
        int startPos = 0;
        int endPos = 0;
        int minDistance = this.expSettings.getParams().getMinLociDistance();
        Object[] loci = this.data.getLoci(chromosome);
        Arrays.sort(loci);
        int i = loci.length - 2;
        while (i >= 0) {
            if (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                startPos = i + 1;
                endPos = i--;
                if (i >= 0) {
                    while (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                        --endPos;
                        if (--i < 0) break;
                    }
                }
                int j = startPos;
                while (j >= endPos) {
                    ((LocusInfo)loci[j]).setFiltered(true);
                    ++removed;
                    --j;
                }
            }
            --i;
        }
        return removed;
    }

    private int[] filterRandomLoci() {
        int[] allRemoved = new int[this.data.getChromosomes()];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        RandomPositionsFilterThread[] randomPositionsFilterThread = new RandomPositionsFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                randomPositionsFilterThread[threadCount] = new RandomPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(randomPositionsFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[randomPositionsFilterThread[j].getChromosome()] = randomPositionsFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                randomPositionsFilterThread[threadCount] = new RandomPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(randomPositionsFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[randomPositionsFilterThread[i2].getChromosome()] = randomPositionsFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).filteringNearLociFinished(this.data, allRemoved);
            ++l;
        }
        return allRemoved;
    }

    private int filterRandomLociInChromosome(int chromosome) {
        if (this.data.getLociCount(chromosome) < 2) {
            return 0;
        }
        int removed = 0;
        int startPos = 0;
        int endPos = 0;
        int minDistance = this.expSettings.getParams().getMinLociDistance();
        Object[] loci = this.data.getLoci(chromosome);
        Arrays.sort(loci);
        Random r = new Random();
        int i = loci.length - 2;
        while (i >= 0) {
            if (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                startPos = i + 1;
                endPos = i--;
                if (i >= 0) {
                    while (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                        --endPos;
                        if (--i < 0) break;
                    }
                }
                int rand = r.nextInt(startPos - endPos + 1) + endPos;
                int j = startPos;
                while (j >= endPos) {
                    if (j != rand) {
                        ((LocusInfo)loci[j]).setFiltered(true);
                        ++removed;
                    }
                    --j;
                }
            }
            --i;
        }
        return removed;
    }

    private int[] filterMiddleLoci() {
        int[] allRemoved = new int[this.data.getChromosomes()];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        MiddlePositionsFilterThread[] middlePositionsFilterThread = new MiddlePositionsFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                middlePositionsFilterThread[threadCount] = new MiddlePositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(middlePositionsFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[middlePositionsFilterThread[j].getChromosome()] = middlePositionsFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                middlePositionsFilterThread[threadCount] = new MiddlePositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(middlePositionsFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[middlePositionsFilterThread[i2].getChromosome()] = middlePositionsFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        int l = 0;
        while (l < this.listeners.size()) {
            this.listeners.get(l).filteringNearLociFinished(this.data, allRemoved);
            ++l;
        }
        return allRemoved;
    }

    private int filterMiddleLociInChromosome(int chromosome) {
        if (this.data.getLociCount(chromosome) < 2) {
            return 0;
        }
        int removed = 0;
        int startPos = 0;
        int endPos = 0;
        int minDistance = this.expSettings.getParams().getMinLociDistance();
        Object[] loci = this.data.getLoci(chromosome);
        Arrays.sort(loci);
        int i = loci.length - 2;
        while (i >= 0) {
            if (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                startPos = i + 1;
                endPos = i--;
                if (i >= 0) {
                    while (((LocusInfo)loci[i + 1]).getCoordinate() - ((LocusInfo)loci[i]).getCoordinate() < minDistance) {
                        --endPos;
                        if (--i < 0) break;
                    }
                }
                int mid = (startPos + endPos) / 2;
                int j = startPos;
                while (j >= endPos) {
                    if (j != mid) {
                        ((LocusInfo)loci[j]).setFiltered(true);
                        ++removed;
                    }
                    --j;
                }
            }
            --i;
        }
        return removed;
    }

    private float[] filterMultipleAssignments() {
        float[] allRemoved = new float[this.data.getChromosomes()];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getPositionsCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        MultipleAssignmentsFilterThread[] multipleAssignmentsFilterThread = new MultipleAssignmentsFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                multipleAssignmentsFilterThread[threadCount] = new MultipleAssignmentsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multipleAssignmentsFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[multipleAssignmentsFilterThread[j].getChromosome()] = multipleAssignmentsFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                multipleAssignmentsFilterThread[threadCount] = new MultipleAssignmentsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multipleAssignmentsFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[multipleAssignmentsFilterThread[i2].getChromosome()] = multipleAssignmentsFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        return allRemoved;
    }

    private float filterMultipleAssignmentsInChromosome(int chromosome) {
        if (this.data.getPositionsCount(chromosome) == 0) {
            return 0.0f;
        }
        float filtered = 0.0f;
        PositionInfo[] positions = this.data.getMatchedPositions(chromosome);
        int i = 0;
        while (i < positions.length) {
            if (positions[i].getLoci2Coordinate() != -1) {
                filtered += positions[i].getTagCount();
                int lociCoord = positions[i].getLociCoordinate();
                this.data.getLocus(chromosome, lociCoord).subtractTagCount(positions[i].getTagCount());
                this.data.filterPosition(chromosome, positions[i].getCoordinate());
            }
            ++i;
        }
        return filtered;
    }

    private float[][] filterLowTagsLoci(int minTagCount, boolean filterPositions) {
        float[][] allRemoved = new float[this.data.getChromosomes()][2];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        LowTagsLociFilterThread[] lowTagsLociFilterThread = new LowTagsLociFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                lowTagsLociFilterThread[threadCount] = new LowTagsLociFilterThread(indexes[i2], minTagCount, filterPositions);
                thread[threadCount] = new Thread(lowTagsLociFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[lowTagsLociFilterThread[j].getChromosome()] = lowTagsLociFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                lowTagsLociFilterThread[threadCount] = new LowTagsLociFilterThread(indexes[i2], minTagCount, filterPositions);
                thread[threadCount] = new Thread(lowTagsLociFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[lowTagsLociFilterThread[i2].getChromosome()] = lowTagsLociFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        return allRemoved;
    }

    private float[] filterLowTagsLociInChromosome(int chromosome, int minTagCount, boolean filterPositions) {
        float[] filtered = new float[2];
        if (this.data.getLociCount(chromosome) == 0) {
            return filtered;
        }
        int minTags = minTagCount;
        Object[] loci = this.data.getLoci(chromosome);
        Object[] positions = this.data.getMatchedPositions(chromosome);
        Arrays.sort(loci);
        Arrays.sort(positions);
        int i = 0;
        while (i < loci.length) {
            if (((LocusInfo)loci[i]).getTagCount() < (float)minTags) {
                int lociCoords = ((LocusInfo)loci[i]).getCoordinate();
                if (filterPositions) {
                    int j = 0;
                    while (j < positions.length) {
                        if (((PositionInfo)positions[j]).getLociCoordinate() == lociCoords) {
                            this.data.filterPosition(chromosome, ((PositionInfo)positions[j]).getCoordinate());
                            filtered[0] = filtered[0] + ((PositionInfo)positions[j]).getTagCount();
                        }
                        ++j;
                    }
                }
                filtered[1] = filtered[1] + 1.0f;
                this.data.filterLocus(chromosome, ((LocusInfo)loci[i]).getCoordinate());
            }
            ++i;
        }
        return filtered;
    }

    private float[][] filterLowTagsPositions() {
        float[][] allRemoved = new float[this.data.getChromosomes()][2];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getPositionsCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        LowTagsPositionsFilterThread[] lowTagsPositionsFilterThread = new LowTagsPositionsFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                lowTagsPositionsFilterThread[threadCount] = new LowTagsPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(lowTagsPositionsFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[lowTagsPositionsFilterThread[j].getChromosome()] = lowTagsPositionsFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                lowTagsPositionsFilterThread[threadCount] = new LowTagsPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(lowTagsPositionsFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[lowTagsPositionsFilterThread[i2].getChromosome()] = lowTagsPositionsFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        return allRemoved;
    }

    private float[] filterLowTagsPositionsInChromosome(int chromosome) {
        float[] filtered = new float[2];
        if (this.data.getPositionsCount(chromosome) == 0) {
            return filtered;
        }
        int minTags = this.expSettings.getParams().getMinPositionTagCount();
        Object[] positions = this.data.getMatchedPositions(chromosome);
        Arrays.sort(positions);
        int i = 0;
        while (i < positions.length) {
            if (((PositionInfo)positions[i]).getTagCount() < (float)minTags) {
                int lociCoords = ((PositionInfo)positions[i]).getLociCoordinate();
                LocusInfo info = this.data.getLocus(chromosome, lociCoords);
                info.subtractTagCount(((PositionInfo)positions[i]).getTagCount());
                if (info.getTagCount() <= 0.0f) {
                    this.data.filterLocus(chromosome, lociCoords);
                }
                int positionCoords = ((PositionInfo)positions[i]).getCoordinate();
                this.data.filterPosition(chromosome, positionCoords);
                filtered[0] = filtered[0] + ((PositionInfo)positions[i]).getTagCount();
                filtered[1] = filtered[1] + 1.0f;
            }
            ++i;
        }
        return filtered;
    }

    private float[][] filterMultiTagsLoci() {
        float[][] allRemoved = new float[this.data.getChromosomes()][2];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getLociCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        MultiTagsLociFilterThread[] multiTagsLociFilterThread = new MultiTagsLociFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                multiTagsLociFilterThread[threadCount] = new MultiTagsLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multiTagsLociFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[multiTagsLociFilterThread[j].getChromosome()] = multiTagsLociFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                multiTagsLociFilterThread[threadCount] = new MultiTagsLociFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multiTagsLociFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[multiTagsLociFilterThread[i2].getChromosome()] = multiTagsLociFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        return allRemoved;
    }

    private float[] filterMultiTagsLociInChromosome(int chromosome) {
        float[] filtered = new float[2];
        if (this.data.getLociCount(chromosome) == 0) {
            return filtered;
        }
        LocusInfo[] loci = this.data.getLoci(chromosome);
        PositionInfo[] positions = this.data.getMatchedPositions(chromosome);
        int maxTags = this.expSettings.getParams().getMaxLociTagCount();
        int i = 0;
        while (i < loci.length) {
            if (loci[i].getTagCount() > (float)maxTags) {
                int lociCoords = loci[i].getCoordinate();
                int j = 0;
                while (j < positions.length) {
                    if (positions[j].getLociCoordinate() == lociCoords) {
                        this.data.filterPosition(chromosome, positions[j].getCoordinate());
                        filtered[0] = filtered[0] + positions[j].getTagCount();
                    }
                    ++j;
                }
                filtered[1] = filtered[1] + 1.0f;
                this.data.filterLocus(chromosome, loci[i].getCoordinate());
            }
            ++i;
        }
        return filtered;
    }

    private float[][] filterMultiTagsPositions() {
        float[][] allRemoved = new float[this.data.getChromosomes()][2];
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getPositionsCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        MultiTagsPositionsFilterThread[] multiTagsPositionsFilterThread = new MultiTagsPositionsFilterThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                multiTagsPositionsFilterThread[threadCount] = new MultiTagsPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multiTagsPositionsFilterThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                        allRemoved[multiTagsPositionsFilterThread[j].getChromosome()] = multiTagsPositionsFilterThread[j].getFiltered();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                multiTagsPositionsFilterThread[threadCount] = new MultiTagsPositionsFilterThread(indexes[i2]);
                thread[threadCount] = new Thread(multiTagsPositionsFilterThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
                allRemoved[multiTagsPositionsFilterThread[i2].getChromosome()] = multiTagsPositionsFilterThread[i2].getFiltered();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
        return allRemoved;
    }

    private float[] filterMultiTagsPositionsInChromosome(int chromosome) {
        float[] filtered = new float[2];
        if (this.data.getPositionsCount(chromosome) == 0) {
            return filtered;
        }
        PositionInfo[] positions = this.data.getMatchedPositions(chromosome);
        int maxTags = this.expSettings.getParams().getMaxPositionTagCount();
        int i = 0;
        while (i < positions.length) {
            if (positions[i].getTagCount() > (float)maxTags) {
                int lociCoords = positions[i].getLociCoordinate();
                LocusInfo info = this.data.getLocus(chromosome, lociCoords);
                info.subtractTagCount(positions[i].getTagCount());
                if (info.getTagCount() <= 0.0f) {
                    this.data.filterLocus(chromosome, lociCoords);
                }
                int positionCoords = positions[i].getCoordinate();
                this.data.filterPosition(chromosome, positionCoords);
                filtered[0] = filtered[0] + positions[i].getTagCount();
                filtered[1] = filtered[1] + 1.0f;
            }
            ++i;
        }
        return filtered;
    }

    private void assignTagsToLoci(int length, int offset) {
        int[] sizes = new int[this.data.getChromosomes()];
        int i = 0;
        while (i < this.data.getChromosomes()) {
            sizes[i] = this.data.getPositionsCount(i);
            ++i;
        }
        int[] indexes = Sort.indexedHeapSort(sizes);
        TagsAssignThread[] tagsAssignThread = new TagsAssignThread[this.cores];
        Thread[] thread = new Thread[this.cores];
        int threadCount = 0;
        int i2 = indexes.length - 1;
        while (i2 >= 0) {
            if (threadCount < this.cores) {
                tagsAssignThread[threadCount] = new TagsAssignThread(indexes[i2], length, offset);
                thread[threadCount] = new Thread(tagsAssignThread[threadCount]);
                thread[threadCount].start();
                ++threadCount;
            } else {
                int j = 0;
                while (j < threadCount) {
                    try {
                        thread[j].join();
                    }
                    catch (Exception e) {
                        Logger.writeExceptionLog(e);
                    }
                    ++j;
                }
                threadCount = 0;
                tagsAssignThread[threadCount] = new TagsAssignThread(indexes[i2], length, offset);
                thread[threadCount] = new Thread(tagsAssignThread[threadCount]);
                thread[threadCount].run();
                ++threadCount;
            }
            --i2;
        }
        i2 = 0;
        while (i2 < threadCount) {
            try {
                thread[i2].join();
            }
            catch (Exception e) {
                Logger.writeExceptionLog(e);
            }
            ++i2;
        }
    }

    private void assignTagsToPositionsInChromosome(int chromosome, int length, int offset) {
        if (this.data.getPositionsCount(chromosome) == 0) {
            return;
        }
        Object[] positions = this.data.getAllPositions(chromosome);
        Object[] loci = this.data.getLoci(chromosome);
        int posLen = loci.length;
        int tagLen = positions.length;
        int toleranceStart = offset - length / 2;
        int toleranceEnd = offset + length / 2;
        int index = 0;
        int startPos = 0;
        int endPos = posLen - 1;
        int dist2left = 0;
        int dist2right = 0;
        Arrays.sort(positions);
        Arrays.sort(loci);
        int tagNum = 0;
        while (tagNum < tagLen) {
            index = this.findIndex(endPos, startPos, ((PositionInfo)positions[tagNum]).getCoordinate(), (LocusInfo[])loci, length, offset);
            if (index != -1 && !((LocusInfo)loci[index]).isFiltered()) {
                startPos = index;
                ((PositionInfo)positions[tagNum]).setLociCoordinate(((LocusInfo)loci[index]).getCoordinate());
                ((PositionInfo)positions[tagNum]).setLociDistance(this.getDistance(((PositionInfo)positions[tagNum]).getCoordinate(), (LocusInfo)loci[index]));
                ((PositionInfo)positions[tagNum]).setMatched(true);
                ((LocusInfo)loci[index]).addTagCount(((PositionInfo)positions[tagNum]).getTagCount());
                ((LocusInfo)loci[index]).setMatched(true);
                dist2left = 0;
                dist2right = 0;
                if (index > 0) {
                    dist2left = this.getDistance(((PositionInfo)positions[tagNum]).getCoordinate(), (LocusInfo)loci[index - 1]);
                }
                if (index < posLen - 1) {
                    dist2right = this.getDistance(((PositionInfo)positions[tagNum]).getCoordinate(), (LocusInfo)loci[index + 1]);
                }
                if ((dist2left != 0 && dist2right == 0 || Math.abs(dist2left) < Math.abs(dist2right)) && dist2left >= toleranceStart && dist2left < toleranceEnd) {
                    ((PositionInfo)positions[tagNum]).setLoci2Coordinate(((PositionInfo)positions[tagNum]).getCoordinate() - dist2left);
                } else if ((dist2left == 0 && dist2right != 0 || Math.abs(dist2left) > Math.abs(dist2right)) && dist2right >= toleranceStart && dist2right < toleranceEnd) {
                    ((PositionInfo)positions[tagNum]).setLoci2Coordinate(dist2right + ((PositionInfo)positions[tagNum]).getCoordinate());
                }
            }
            ++tagNum;
        }
    }

    private int findIndex(int endPos, int startPos, int searchedTag, LocusInfo[] loci, int length, int offset) {
        int mid = 0;
        int distance = 0;
        int index = 0;
        int posStart = offset - length / 2;
        int posEnd = offset + length / 2;
        while (startPos <= endPos) {
            mid = (startPos + endPos) / 2;
            distance = this.getDistance(searchedTag, loci[mid]);
            if (!loci[mid].isFiltered() && distance >= posStart && distance < posEnd) {
                distance = Math.abs(distance);
                index = mid - 1;
                while (index >= 0 && Math.abs(loci[index].getCoordinate() - searchedTag) <= distance && this.getDistance(searchedTag, loci[index]) >= posStart && this.getDistance(searchedTag, loci[index]) < posEnd) {
                    mid = index;
                    distance = Math.abs(loci[index].getCoordinate() - searchedTag);
                    --index;
                }
                index = mid + 1;
                while (index <= endPos && Math.abs(loci[index].getCoordinate() - searchedTag) <= distance && this.getDistance(searchedTag, loci[index]) >= posStart && this.getDistance(searchedTag, loci[index]) < posEnd) {
                    mid = index;
                    distance = Math.abs(loci[index].getCoordinate() - searchedTag);
                    ++index;
                }
                return mid;
            }
            if (loci[mid].getCoordinate() > searchedTag) {
                endPos = mid - 1;
                continue;
            }
            startPos = mid + 1;
        }
        return -1;
    }

    public int getDistance(int tag, LocusInfo locus) {
        if (!this.expSettings.getParams().isCheckStrand() || locus.getStrand() == 1) {
            return tag - locus.getCoordinate();
        }
        return locus.getCoordinate() - tag;
    }

    public void addListener(ProfilerListener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(ProfilerListener listener) {
        this.listeners.remove(listener);
    }

    @Override
    public void lineRead(int fileLengthLeft) {
        int i = 0;
        while (i < this.listeners.size()) {
            this.listeners.get(i).tagsReadingUpdate(fileLengthLeft);
            ++i;
        }
    }

    class AllLociFilterThread
    implements Runnable {
        private int chromosome;
        private int filtered;

        public AllLociFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterAllLociInChromosome(this.chromosome);
        }

        public int getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class LowTagsLociFilterThread
    implements Runnable {
        private int chromosome;
        private float[] filtered;
        private int minTagCount;
        private boolean filterPositions;

        public LowTagsLociFilterThread(int chromosome, int minTagCount, boolean filterPositions) {
            this.chromosome = chromosome;
            this.minTagCount = minTagCount;
            this.filterPositions = filterPositions;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterLowTagsLociInChromosome(this.chromosome, this.minTagCount, this.filterPositions);
        }

        public float[] getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class LowTagsPositionsFilterThread
    implements Runnable {
        private int chromosome;
        private float[] filtered;

        public LowTagsPositionsFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterLowTagsPositionsInChromosome(this.chromosome);
        }

        public float[] getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class MiddlePositionsFilterThread
    implements Runnable {
        private int chromosome;
        private int filtered;

        public MiddlePositionsFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterMiddleLociInChromosome(this.chromosome);
        }

        public int getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class MultiTagsLociFilterThread
    implements Runnable {
        private int chromosome;
        private float[] filtered;

        public MultiTagsLociFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterMultiTagsLociInChromosome(this.chromosome);
        }

        public float[] getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class MultiTagsPositionsFilterThread
    implements Runnable {
        private int chromosome;
        private float[] filtered;

        public MultiTagsPositionsFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterMultiTagsPositionsInChromosome(this.chromosome);
        }

        public float[] getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class MultipleAssignmentsFilterThread
    implements Runnable {
        private int chromosome;
        private float filtered;

        public MultipleAssignmentsFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterMultipleAssignmentsInChromosome(this.chromosome);
        }

        public float getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class RandomPositionsFilterThread
    implements Runnable {
        private int chromosome;
        private int filtered;

        public RandomPositionsFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterRandomLociInChromosome(this.chromosome);
        }

        public int getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class ShortLociFilterThread
    implements Runnable {
        private int chromosome;
        private int filtered;

        public ShortLociFilterThread(int chromosome) {
            this.chromosome = chromosome;
        }

        @Override
        public void run() {
            this.filtered = Profiler.this.filterShortLociInChromosome(this.chromosome);
        }

        public int getFiltered() {
            return this.filtered;
        }

        public int getChromosome() {
            return this.chromosome;
        }
    }

    class TagsAssignThread
    implements Runnable {
        private int chromosome;
        private int length;
        private int offset;

        public TagsAssignThread(int chromosome, int length, int offset) {
            this.chromosome = chromosome;
            this.length = length;
            this.offset = offset;
        }

        @Override
        public void run() {
            Profiler.this.assignTagsToPositionsInChromosome(this.chromosome, this.length, this.offset);
        }
    }
}

